<?php

/* --------------------------------------------------------------
   StyleEditProductSearchController.inc.php 2019-10-02
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2016 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

/**
 * Class representing a product search controller for StyleEdit
 */
class StyleEditProductSearchController extends HttpViewController
{
    /**
     * Product reader service
     *
     * @var ProductReadService
     */
    protected $productReadService;
    
    /**
     * Product list item JSON serializer
     *
     * @var ProductListItemJsonSerializer
     */
    protected $productListItemJsonSerializer;
    
    
    /**
     * Initialize
     */
    public function init()
    {
        $this->productReadService            = StaticGXCoreLoader::getService('ProductRead');
        $this->productListItemJsonSerializer = MainFactory::create(ProductListItemJsonSerializer::class);
    }
    
    
    /**
     * Return the search result using the provided search term
     *
     * @return JsonHttpControllerResponse
     */
    public function actionByTerm()
    {
        $result = [];
        
        $term     = $this->_getQueryParameter('term');
        $language = $this->_getQueryParameter('language');
        
        if ($term === null || $language === null) {
            throw new InvalidArgumentException('Invalid or missing parameters. Required parameters: "term" and "language"');
        }
        
        $languageCode = MainFactory::create(LanguageCode::class, new StringType($language));
        
        $searchCondition = ProductSearchCondition::createByArray([
                                                                     'search' => [
                                                                         'should' => [
                                                                             [
                                                                                 'match' => [
                                                                                     'products.products_id' => $term
                                                                                 ]
                                                                             ],
                                                                             [
                                                                                 'like' => [
                                                                                     'products_description.products_name' => "%{$term}%"
                                                                                 ]
                                                                             ]
                                                                         ]
                                                                     ]
                                                                 ]);
        
        $products = $this->productReadService->searchProducts($languageCode, $searchCondition)->getArray();
        
        foreach ($products as $product) {
            $data     = json_decode($this->productListItemJsonSerializer->serialize($product));
            $result[] = [
                'id'     => $data->id,
                'name'   => $data->name,
                'active' => $data->isActive,
                'image'  => $this->getCorrectImagePath($data->image)
            ];
        }
        
        return MainFactory::create(JsonHttpControllerResponse::class, $result);
    }
    
    
    /**
     * Return the search result using a list of product IDs
     *
     * @return JsonHttpControllerResponse
     */
    public function actionByIds()
    {
        $result = [];
        
        $ids      = $this->_getQueryParameter('ids');
        $language = $this->_getQueryParameter('language');
        
        if ($ids === null || $language === null) {
            throw new InvalidArgumentException('Invalid or missing parameters. Required parameters: "ids" and "language"');
        }
        
        $languageCode = MainFactory::create(LanguageCode::class, new StringType($language));
        $idCollection = explode(',', $ids);
        
        foreach ($idCollection as $id) {
            $parsedId = new IdType((int)$id);
            
            try {
                $product = $this->productReadService->getProductById($parsedId);
                
                $result[] = [
                    'id'     => $product->getProductId(),
                    'name'   => $product->getName($languageCode),
                    'active' => $product->isActive(),
                    'image'  => $this->getCorrectImagePath($product->getPrimaryImage()->getFilename())
                ];
            } catch (Exception $exception) {
                continue;
            }
        }
        
        return MainFactory::create(JsonHttpControllerResponse::class, $result);
    }
    
    
    /**
     * @param $image
     *
     * @return string
     */
    protected function getCorrectImagePath($image): string
    {
        return (ENABLE_SSL_CATALOG === 'true' ? HTTPS_CATALOG_SERVER : HTTP_CATALOG_SERVER) . DIR_WS_CATALOG
               . "images/product_images/thumbnail_images/{$image}";
    }
}
